<?php
namespace mail\jianmail;

use app\admin\lib\Plugin;

class JianmailPlugin extends Plugin
{
    # 基础信息（必须按规范填写）
    public $info = array(
        'name'        => 'Jianmail', // 插件类名不带Plugin
        'title'       => '简讯邮箱',
        'description' => '简讯个人邮箱API接口',
        'status'      => 1,
        'author'      => '简讯',
        'version'     => '1.0',
        'help_url'    => 'https://email.uvqzu.cn/', // 申请接口地址
    );

    # 插件安装（无逻辑操作返回true）
    public function install()
    {
        return true;
    }

    # 插件卸载（无逻辑操作返回true）
    public function uninstall()
    {
        return true;
    }

    /**
     * 邮件发送函数（核心接口）
     * @param array $params 系统传入参数
     * @return array 必须返回指定格式
     */
    public function send($params)
    {
        try {
            // 获取配置项
            $config = $params['config'];
            
            // 验证必填参数
            if (empty($config['systememail']) || empty($config['password'])) {
                return ['status' => 'error', 'msg' => '配置信息不完整'];
            }
            
            if (empty($params['email'])) {
                return ['status' => 'error', 'msg' => '收件人邮箱不能为空'];
            }

            // 构建API请求数据（按照文档规范）
            $postData = [
                'from'      => $config['systememail'],  // 发件人邮箱
                'password'  => $config['password'],     // 邮箱密码
                'to'        => $params['email'],       // 收件人邮箱
                'subject'   => $params['subject'] ?? '', // 邮件标题
                'content'   => $params['content'] ?? '', // 邮件内容
                'type'      => 'html'                   // 邮件类型（默认html）
            ];

            // 固定API地址（文档中指定的地址）
            $apiUrl = 'https://email.uvqzu.cn/send_api.php';

            // 发送HTTP请求
            $result = $this->httpPost($apiUrl, $postData);
            
            // 解析JSON响应
            $response = json_decode($result, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                return ['status' => 'error', 'msg' => 'API响应格式错误: ' . $result];
            }

            // 判断API返回状态（根据文档，成功返回success: true）
            if (isset($response['success']) && $response['success'] === true) {
                return ['status' => 'success'];
            } else {
                $errorMsg = $response['message'] ?? '邮件发送失败';
                return ['status' => 'error', 'msg' => $errorMsg];
            }

        } catch (\Exception $e) {
            return ['status' => 'error', 'msg' => '系统异常: ' . $e->getMessage()];
        }
    }

    /**
     * HTTP POST请求封装（私有方法）
     * 使用JSON格式发送数据（与文档一致）
     */
    private function httpPost($url, $data)
    {
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => json_encode($data), // 使用JSON格式
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_HTTPHEADER     => [
                'Content-Type: application/json', // JSON请求头
                'User-Agent: Jianmail-Client/1.0'
            ]
        ]);
        
        $response = curl_exec($ch);
        $error = curl_error($ch);
        $errno = curl_errno($ch);
        
        curl_close($ch);
        
        if ($errno) {
            throw new \Exception("请求失败: {$error}");
        }
        
        return $response;
    }
}
